/*++

Copyright (C) 2009 Giovanni Panozzo All Rights Reserved
based on code of lhid2hci.c and hid2hic of linux udev package
Copyright (C) 2006 Ken Johnson (Skywing) All Rights Reserved

Module Name:

	lhid2hci.c

Abstract:

	Converts some Dell Bluetooth cards from HID emulation mode
	to native HCI mode.

Author: Giovanni Panozzo

Environment:

	Win32 console application

Revision History:

--*/

#define WIN32_NO_STATUS


#include <windows.h>
#include <ddk/ntstatus.h>
#include <ddk/hidusage.h>
#include <ddk/hidsdi.h>
#include <ddk/hidpi.h>
#include <ddk/hidclass.h>
#include <SetupAPI.h>
#include <stdlib.h>
#include <stdio.h>
#include <tchar.h>

#ifndef NT_SUCCESS
#define NT_SUCCESS(Status) ((NTSTATUS)(Status) >= 0)
#endif

USHORT g_TargetVid;
USHORT g_TargetPid;

// This is not included in cygwin hidsdi.h
WINHIDSDI BOOL WINAPI HidD_SetOutputReport(HANDLE, PVOID, ULONG);

struct req_vendor_t {
	unsigned int timeout;
	unsigned int type;
	unsigned int recipient;
	unsigned int request;
	unsigned int value;
	unsigned int index;
};


void prError(TCHAR *lpszFunction) 
{ 
    // Retrieve the system error message for the last-error code

    LPVOID lpMsgBuf;
    DWORD dw = GetLastError(); 

    FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER | 
        FORMAT_MESSAGE_FROM_SYSTEM |
        FORMAT_MESSAGE_IGNORE_INSERTS,
        NULL,
        dw,
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
        (LPTSTR) &lpMsgBuf,
        0, NULL );

    // Display the error message and exit the process

    _tprintf(TEXT("%s failed with error %u: %s"),
        lpszFunction, (unsigned int)dw, (TCHAR *)lpMsgBuf); 

    LocalFree(lpMsgBuf);

}

BOOLEAN SendDellHidOutputReport(
	HANDLE HidDeviceObject,
	PCHAR dt,
	USHORT dtcount,
	USHORT VendorID,
	USHORT ProductID
	)
{
	PHIDP_PREPARSED_DATA PreparsedData;
	BOOLEAN              Success;
	DWORD bret;

	Success = FALSE;

	if (!HidD_GetPreparsedData(	HidDeviceObject, &PreparsedData))
	{
		wprintf(
			L"Failed to get preparsed data.\n");
		return FALSE;
	}

	for (;;)
	{
		HIDD_ATTRIBUTES HidAttributes;
		HIDP_CAPS       HidCaps;

		HidAttributes.Size = sizeof(HIDD_ATTRIBUTES);

		if (!HidD_GetAttributes( HidDeviceObject, &HidAttributes) )	{
			wprintf(L"Failed to get HID attributes.\n");
			break;
		}

		wprintf( L"Found HID device (%04X, %04X) ",HidAttributes.VendorID, HidAttributes.ProductID);

		if (HidAttributes.VendorID != VendorID || HidAttributes.ProductID != ProductID)	{
			wprintf( L"skipping.\n");
			break;
		}

		
		if (!HidP_GetCaps( PreparsedData, &HidCaps)) {
			wprintf( L"Failed to get HID capabilities.\n");
			break;
		}
		
		
		/* _tprintf(TEXT("HID_CAPS.InputReportByteLength = %u\n"), (unsigned int)HidCaps.InputReportByteLength );
		_tprintf(TEXT("HID_CAPS.OutputReportByteLength = %u\n"), (unsigned int)HidCaps.OutputReportByteLength );
		_tprintf(TEXT("HID_CAPS.FeatureReportByteLength = %u\n"), (unsigned int)HidCaps.FeatureReportByteLength );
		*/

		if ( !DeviceIoControl(HidDeviceObject, IOCTL_HID_SET_FEATURE, dt, dtcount, NULL, 0, &bret, NULL) ) {
			prError(TEXT("DeviceIoControl"));
			break;
		}
		
		Success = TRUE;
		break;
	}


	return Success;
}

BOOLEAN SendReportsToDevicePath(
	IN CONST TCHAR* DevicePath
	)
{
	HANDLE  HidDeviceObject;
	BOOLEAN Success = FALSE;

	// _tprintf(TEXT("Opening file %s\n"),DevicePath);
	
	HidDeviceObject = CreateFile(DevicePath,0,FILE_SHARE_READ | FILE_SHARE_WRITE,0,	OPEN_EXISTING,	0,	0);

	if (HidDeviceObject == INVALID_HANDLE_VALUE)
	{
		_tprintf(TEXT("Failed to open hid device: %s (%lu)\n"),	DevicePath, GetLastError());
		return FALSE;
	}

	for (;;)
	{
		CHAR Report[4] = { 0x7f, 0x13, 0x00, 0x00 };
		
		
	
		if (SendDellHidOutputReport(HidDeviceObject,Report,sizeof(Report),g_TargetVid,g_TargetPid))
		{
			_tprintf(TEXT("Sent report to device %s.\n"),	DevicePath);
			Success = TRUE;
		}

		break;
	}

	CloseHandle(HidDeviceObject);

	return Success;
}

ULONG SendReportsToDeviceInfoList(
	HDEVINFO DevInfo,
	LPGUID DeviceClassGuid
	)
{
	PSP_DEVICE_INTERFACE_DETAIL_DATA DeviceInterfaceDetailData = 0;
	SP_DEVICE_INTERFACE_DATA         DeviceInterfaceData;
	ULONG                            RequiredSize;
	ULONG                            MemberIndex = 0;
	ULONG                            ErrorCode;
	ULONG                            Converted = 0;

	DeviceInterfaceData.cbSize = sizeof(SP_DEVICE_INTERFACE_DATA);

	while (SetupDiEnumDeviceInterfaces(
		DevInfo,
		0,
		DeviceClassGuid,
		MemberIndex++,
		&DeviceInterfaceData))
	{
		DeviceInterfaceData.cbSize = sizeof(SP_DEVICE_INTERFACE_DATA);

		if (DeviceInterfaceDetailData)
		{
			HeapFree( GetProcessHeap(),	0, DeviceInterfaceDetailData);
			DeviceInterfaceDetailData = 0;
		}

		if (SetupDiGetDeviceInterfaceDetail( DevInfo, &DeviceInterfaceData, 0, 0, &RequiredSize, 0) || GetLastError() != ERROR_INSUFFICIENT_BUFFER)
		{
			wprintf( L"Failed to query device interface detail data length for device %lu.\n",MemberIndex-1);
			continue;
		}

		DeviceInterfaceDetailData = (PSP_DEVICE_INTERFACE_DETAIL_DATA)HeapAlloc( GetProcessHeap(), 0, RequiredSize);

		if (!DeviceInterfaceDetailData)
		{
			wprintf( L"Failed to allocate memory for device interface detail data for device %lu (%lu bytes).\n",
				MemberIndex-1, RequiredSize);
			continue;
		}

		DeviceInterfaceDetailData->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);

		if (!SetupDiGetDeviceInterfaceDetail( DevInfo, &DeviceInterfaceData, DeviceInterfaceDetailData, RequiredSize, &RequiredSize, 0))
		{
			wprintf( L"Failed to query device interface detail data for device %lu: %lu.",
				MemberIndex-1, GetLastError());
			continue;
		}

//		wprintf( L"\nFound device interface: %s\n",	DeviceInterfaceDetailData->DevicePath);
		
		if (SendReportsToDevicePath( DeviceInterfaceDetailData->DevicePath))
			Converted++;
	}

	ErrorCode = GetLastError();

	if (ErrorCode != ERROR_NO_MORE_ITEMS)
	{
		wprintf(
			L"Failed to enumerate device interfaces: %lu\n", GetLastError());
	}

	if (DeviceInterfaceDetailData)
		HeapFree(
			GetProcessHeap(),
			0,
			DeviceInterfaceDetailData);

	return Converted;
}

ULONG SendReportsToDeviceClass(
	LPGUID DeviceClassGuid
	)
{
	HDEVINFO DevInfo;
	ULONG    Converted;

	DevInfo = SetupDiGetClassDevs(
		DeviceClassGuid,
		0,
		0,
		DIGCF_PRESENT|DIGCF_DEVICEINTERFACE);

	if (DevInfo == (HDEVINFO)INVALID_HANDLE_VALUE)
	{
		wprintf(L"Failed to enumerate devices.\n");
		return 0;
	}

	Converted = SendReportsToDeviceInfoList(
		DevInfo,
		DeviceClassGuid);

	SetupDiDestroyDeviceInfoList(
		DevInfo);

	return Converted;
}

int main(int argc, TCHAR *argv[])
{
	GUID  HidGuid;
	ULONG Converted;

	_tprintf(TEXT("DHid2Hci v 0.1 - Copyright (C) 2009 Giovanni Panozzo.\nBased on code from lhid2hci by Ken Johnson.\n"
		"Converts Dell Truemobile 370 cards from HID emulation mode to native HCI mode.\n\n"));
		
	if (argc != 1 && argc != 3)
	{
		wprintf(
			L"Usage: dhid2hci <vid> <pid>\n"
			L"Ex: dhid2hci 413C 8158  (for Truemobile 370)\n"
			);

		return 1;
	}

	if ( argc == 3 ) {
		g_TargetVid = (USHORT)_tcstoul(argv[1], 0, 16);
		g_TargetPid = (USHORT)_tcstoul(argv[2], 0, 16);
	} else {
		g_TargetVid = 0x413C;
		g_TargetPid = 0x8158;
	}
	

	wprintf(L"Operating with PnP Vendor ID %04X, Product ID %04X...\n",	g_TargetVid, g_TargetPid);

	HidD_GetHidGuid(&HidGuid);

	Converted = SendReportsToDeviceClass(&HidGuid);

	wprintf(L"Converted %lu HID devices to native HCI mode.\n",	Converted);

	return 0;
}
